#! /usr/bin/python3
# Test program for module {hotpath}.
# Last edited on 2021-02-19 15:51:18 by jstolfi

import hotpath
import path
import block
import example_path
import input_data
import hacks
import job_parms
import rn
import pyx
import sys
from math import sqrt, sin, cos, floor, ceil, inf, nan, pi

import move
import contact
import gcode
import os

parms = job_parms.typical()
#parms = job_parms.very_slow()
oprefix = "tests/out/hotpath_TST"

job_parms.write(sys.stderr, None, parms, None)

def plot_solution(Q, name):
  # Plots the oriened path {Q} to files called "hotpath_TST_{name}.{ext}",
  # where {ext} is "eps", "jpg", "png", etc.
  c = pyx.canvas.canvas()
  
  # Find the bounding box:
  Pbox = path.bbox(Q)

  plo = (floor(Pbox[0][0])-1, floor(Pbox[0][1])-1)
  phi = (ceil(Pbox[1][0])+1, ceil(Pbox[1][1])+1)
  xsz = phi[0] - plo[0]
  ysz = phi[1] - plo[1]

  hacks.plot_frame(c, pyx.color.rgb.white, 0.02, plo, xsz, ysz, 0.01)
  hacks.plot_grid(c, pyx.color.rgb(0.800, 0.750, 0.600), 0.03, plo, xsz, ysz, 0.25, 1,1)
  pyx.unit.set(uscale=1.00, wscale=1.00, vscale=1.00)
  
  ctrace =  pyx.color.rgb(0.050, 0.850, 0.000) # Color of nominal trace area.
  waxes = 0.05*parms['solid_raster_width']; 
  axes = False
  dots = False
  arrows = False
  matter = True
  path.plot_standard(c, Q, (0,0), None, ctrace, waxes, axes, dots, arrows, matter)
  hacks.write_plot(c, "tests/out/hotpath_TST_sol_" + name)
  gcode.write(open("tests/out/hotpath_TST_sol_" + name + '.gcode', 'w+'), Q, 0, parms)
  return 
  # ----------------------------------------------------------------------

def test_tiny(tag):
  # Three blocks, each with one path and its reversal.
  # The paths are two horizontal raster lines and 
  # a bent 2-trace path, roughly horizontal too.  
  #
  # There are no cooling or max-calls constraints. The starting point is
  # fixed at {(1,1)}. The file name will have "_test1" appended.
  sys.stderr.write("----------------------------------------------------------------------\n")
  name = tag
  sys.stderr.write("--- %s ---\n" % name)
  p11 = (2, 2)
  p12 = (5, 2)
  
  p21 = (2, 3)
  p22 = (6, 3)
  
  p31 = (3, 4)
  p32 = (4, 5)
  p33 = (5, 4)
  
  xsz = 7
  ysz = 7
  
  wdf = parms['solid_raster_width']
  
  ph1 = path.from_points((p11, p12,), wdf, parms)
  ph2 = path.from_points((p21, p22,), wdf, parms)
  ph3 = path.from_points((p31, p32, p33,), wdf, parms)
  
  bc1 = block.from_paths((ph1, path.rev(ph1),))
  bc2 = block.from_paths((ph2, path.rev(ph2),))
  bc3 = block.from_paths((ph3, path.rev(ph3),))
  
  o = (1,1)
  BS = (bc2,bc3,bc1,)
  CS = ()
  input_data.plot(oprefix + "_inp_" + name, BS, CS, o, wdf)
  
  Delta = inf
  maxcalls = None
  
  Q = hotpath.best_path(o, BS, CS, Delta, maxcalls, parms)
  
  plot_solution(Q, name)  
  return 
  # ----------------------------------------------------------------------

def test_road_of_rasters(n, o, tag):
  # The input is a single road of {n}, raster lines, each line being a single 
  # block with two choices (the path and its reversal).
  #
  # There are no cooling or max-calls constraints. The starting point is
  # the given point {o}. The file name will have "_test2_{tag}" appended.
  sys.stderr.write("----------------------------------------------------------------------\n")
  name = "%s_%02d" % (tag, n)
  sys.stderr.write("--- %s ---\n" % name)
  
  wdf = parms['solid_raster_width']
  sys.stderr.write("wdf = %.3f mm\n" % wdf)
  
  BS = []
  for k in range(n):
    p = (2, 1+k*wdf)
    q = (4, 1+k*wdf)
    ph = path.from_points((p, q,), wdf, parms)
    bc = block.from_paths((ph, path.rev(ph),))
    BS.append(bc)
    
  CS = ()

  input_data.plot(oprefix + "_inp_" + name, BS, CS, o, wdf)

  Delta = inf
  maxcalls = None
  
  Q = hotpath.best_path(o, BS, CS, Delta, maxcalls, parms)
  
  plot_solution(Q, name)  
  return 
  # ----------------------------------------------------------------------

def test_matrix_of_roads(nx, ny, nr, o, tag):
  # The input has an array of {nx} by {ny} blocks. Each block is a serpentine
  # path of {nr} horizontal rasters and {nr-1} links, in the four orders and directions. 
  #
  # There are no cooling or max-calls constraints. The starting point is
  # the given point {o}. The file name will have "_test3_{tag}" appended.
  sys.stderr.write("----------------------------------------------------------------------\n")
  name = "%s_%02dx_%02d_%02d" % (tag,nx,ny,nr)
  sys.stderr.write("--- %s ---\n" % name)
  
  wdf = parms['solid_raster_width']
  xszr = 3   # Length of raster lines.
  sys.stderr.write("wdf = %.3f mm\n" % wdf)
  
  szb = (xszr, (nr-1)*wdf) # Size of each block, not counting trace widths
  spcb = (2.0*wdf, 1)  # Extra spacing between blocks.
  org =(2,2) # Lower left corner of lower block.
  
  BS = []
  for ixb in range(nx):
    for iyb in range(ny):
      # Compute the lower left corner {orgb} of block.
      orgb = rn.add(org, rn.scale((ixb,iyb), rn.add(szb, spcb))) 
      # Create a list {PS} of the four block choices:
      PS = [] # Choices for the block.
      for od in range(2):
        # Create a path of alternating rasters with order {od}:
        use_jumps = False
        ystep = (1 - 2*od)*wdf
        ph = example_path.raster_rectangle(nr, orgb, xszr, ystep, wdf, use_jumps, parms)
        PS.append(ph)
        PS.append(path.rev(ph))
      bc = block.from_paths(PS)
      BS.append(bc)

  CS = ()

  input_data.plot(oprefix + "_inp_" + name, BS, CS, o, wdf)

  Delta = inf
  maxcalls = None
  
  Q = hotpath.best_path(o, BS, CS, Delta, maxcalls, parms)
  
  plot_solution(Q, name)  
  return 
  # ----------------------------------------------------------------------

def test_two_roads_and_islands(nmv, nmg, nis, o, Delta, maxcalls, tag):
  # The input is two "roads", each with {nmv} rasters grouped into blocks of {nmg} rasters each, and {nis}
  # "islands" at left, between, and at right of the roads
  #
  # There are constraints between successive rasters. The starting point is
  # the given point {o}. The file name will have "_test4_{nmv}_{nis}_{tag}" appended.
  sys.stderr.write("----------------------------------------------------------------------\n")
  name = "%s_%02d_%02d_%02d" % (tag,nmv,nmg,nis)
  sys.stderr.write("--- %s ---\n" % name)
  
  wdf = 0.50
  wdc = wdf/2

  BS, CS = input_data.two_roads_and_islands(nmv, nmg, nis, wdc, wdf, parms)

  input_data.plot(oprefix + "_inp_" + name, BS, CS, o, wdf)
    
  Q = hotpath.best_path(o, BS, CS, Delta, maxcalls, parms)
  if Q != None:
    plot_solution(Q, name)  
  return 
  # ----------------------------------------------------------------------

################################
def isEqual (A, B, epsilon):
  if (abs(A[0] - B[0]) < epsilon) and (abs(A[1] - B[1]) < epsilon):
    return True 
  return False

def create_stroke (sid, p, q, rbit, group):
  return {
      'sid': sid, 
      'p': [p, q],
      'rbit': rbit,
      'group': group,
      'links': [[].copy(),[].copy()]
  }

def add_side (R1, R2, rasterLink0, rasterLink1):
  if not parms['use_rasterlinks']:
    rasterLink0 = None
    rasterLink1 = None

  y1 = R1['p'][0][1]
  y2 = R2['p'][0][1]

  if y2 > y1:
    R1['links'][1].append([R2['sid'], rasterLink0, rasterLink1]) 
    R2['links'][0].append([R1['sid'], rasterLink0, rasterLink1]) 

  elif y1 > y2:
    R1['links'][0].append([R2['sid'], rasterLink0, rasterLink1]) 
    R2['links'][1].append([R1['sid'], rasterLink0, rasterLink1]) 

  return

def read_rasterLink (link):
  if 'None' in link:
    rasterLink = None
      
  else:
    aux = link.split(';')
    rasterLink = list()

    for index in range(1, len(aux)):
      p = aux[index].split('&')
      rasterLink.append((float(p[0]), float(p[1])))

  return rasterLink

def read_file (name):
  R = []
  S = []
  groupMax = 0

  with open('./tests/in/' + name + '.txt', 'r', encoding = "ISO-8859-1") as f:
    for line in f:
      if line[0] == 'N':
        R = [None]*int(line[1:])
      
      elif line[0] == 'R':
        r = line[1:].split(",")
        p_x = float(r[1])
        p_y = float(r[2])

        q_x = float(r[3])
        q_y = float(r[4])

        if p_y != q_y: 
          if(p_y > q_y):
            q_y = p_y
          else:
            p_y = q_y

        if p_x > q_x: 
          x_aux = q_x
          q_x = p_x
          p_x = x_aux

        p = (p_x, p_y)
        q = (q_x, q_y)

        if groupMax < int(r[6]):
          groupMax = int(r[6])

        R[int(r[0])] = create_stroke(int(r[0]), p, q, int(r[5]), int(r[6]))
      
      elif line[0] == 'L':
        l = line[1:].split(",")

        rasterLink0 = read_rasterLink(l[2])
        rasterLink1 = read_rasterLink(l[3])

        add_side(R[int(l[0].replace('L', ''))], R[int(l[1].replace('L', ''))], rasterLink0, rasterLink1)
        S.append((int(l[0].replace('L', '')), int(l[1].replace('L', ''))))

  return R, S, groupMax

def split_blocks(R, groupMax): 
  totalOriginal = groupMax + 1

  for i in range(0, totalOriginal):
    swap_later = False
    group = i
  
    for r in R:
      if swap_later == True:
        swap_later = False
        group = groupMax + 1

      if r['group'] == i:
        input_lines = 0
        output_lines = 0

        if len(r['links'][0]) > 1 or len(r['links'][1]) > 1:
          input_lines = len(r['links'][0])
          output_lines = len(r['links'][1])

          if input_lines > 1:
            group = groupMax + 1

          if output_lines > 1:
            swap_later = True

        r['group'] = group
        
        if r['group'] > groupMax:
          groupMax = r['group']

  return groupMax

def limit_blocks(R, groupMax, maxLimit):
  for i in range(0, groupMax + 1):
    group = i
    lines = 0

    for r in R:
      if r['group'] == i:
        if lines < maxLimit:
          lines = lines + 1
        else:
          lines = 1
          groupMax += 1
          group = groupMax

        r['group'] = group

  return groupMax

def delete_rasterLink(R):
  for index1 in range(len(R)):
    for edge in range(2):
      for indexLink in range(len(R[index1]['links'][edge])):
        index2 = R[index1]['links'][edge][indexLink][0]
        
        if R[index1]['group'] != R[index2]['group']:
          R[index1]['links'][edge][indexLink][1] = None
          R[index1]['links'][edge][indexLink][2] = None

          for indexLink2 in range(len(R[index2]['links'][1-edge])):
            if R[index2]['links'][1-edge][indexLink2][0] == index1:
              R[index2]['links'][1-edge][indexLink2][1] = None
              R[index2]['links'][1-edge][indexLink2][2] = None

  return

def raster_raster_contact(bc0, bc1, parms):
  # The {X} range of the contact will be {[xlo _ xhi]}
  xlo = -inf 
  xhi = +inf
  mv = [None,None]    # Top raster traces at top of {bc0} and bottom of {bc1}.
  for k in range(2):
    bck = (bc0, bc1)[k][0]
    # Pick some choice of the block, {bc0}  or {bc1}
    ophk = block.choice(bck, 0)
    omvk = path.elem(ophk, (bc0, bc1)[k][1])
    
    # Check if they are both traces of the same width
    wdk = move.width(omvk)
    assert wdk > 0 # Must be trace not jump.
    # Save the unoriented move into {mv[k]}:
    mv[k], drk = move.unpack(omvk)
    # Find the {X} span and intersect with current {[xlo _ xhi]}
    pk = move.pini(mv[k])
    qk = move.pfin(mv[k])
    assert pk[1] == qk[1] # Trace must be horizontal.
    xlok = min(pk[0], qk[0])
    xhik = max(pk[0], qk[0])
    xlo = max(xlo, xlok)
    xhi = min(xhi, xhik)

  # Check coordinates:
  y0 = move.pini(mv[0])[1]; wd0 = move.width(mv[0])
  y1 = move.pini(mv[1])[1]; wd1 = move.width(mv[1])
  assert abs((y0 + wd0/2) - (y1 - wd1/2)) < 0.01*(wd0+wd1), "traces are not adjacent"
  assert xlo < xhi, "{X} randges do not overlap"
  
  # Create the contact:
  ymd = (y0 + y1 + (wd0 - wd1)/2)/2
  ct = contact.make((xlo,ymd), (xhi, ymd), mv[0], mv[1], parms)
  return ct

def create_input (name, wdf):
  R, S, groupMax = read_file(name)
  
  if parms['splitLines']:
    groupMax = split_blocks(R, groupMax)
  if parms['limitLines']:
    groupMax = limit_blocks(R, groupMax, parms['nMaxLines'])
  if parms['splitLines'] or parms['limitLines']:
    delete_rasterLink(R)

  BS = []
  CS = []

  for g in range(groupMax + 1):
    points = list()

    for r in R:
      if r['group'] == g:
        if len(points) > 0:
          rasterLink = None

          for link in r['links'][0]:
            if r['group'] == R[link[0]]['group']:
              if link[1 + r['rbit']] != None:
                if isEqual(link[1 + r['rbit']][0], r['p'][r['rbit']], 0.05) or isEqual(link[1 + r['rbit']][-1], r['p'][r['rbit']], 0.05):
                  if isEqual(link[1 + r['rbit']][0], points[-1], 0.05) or isEqual(link[1 + r['rbit']][-1], points[-1], 0.05):
                    rasterLink = link[1 + r['rbit']]
          
          if rasterLink != None:
            if isEqual(rasterLink[0], points[-1], 0.05):
              indexBegin = 0
              indexEnd = len(rasterLink)
              inc = 1
                
            elif isEqual(rasterLink[0], r['p'][r['rbit']], 0.05):
              indexBegin = len(rasterLink) - 1
              indexEnd = -1
              inc = -1    

            for index in range(indexBegin, indexEnd, inc):
              p = rasterLink[index]
              if not isEqual(p, points[-1], 0.05) and not isEqual(p, r['p'][r['rbit']], 0.05):
                points.append(p)

        points.append(r['p'][r['rbit']])
        points.append(r['p'][1-r['rbit']])
      
    if len(points) > 0:
      ph = path.from_points(points, wdf, parms)
      bc = block.from_paths([ph, path.rev(ph)])
      BS.append(bc)
  
  for s in S:
    r0 = R[s[0]]
    r1 = R[s[1]]

    if r0['group'] != r1['group']:
      p0 = r0['p'][r0['rbit']]
      q0 = r0['p'][1-r0['rbit']]
      bc0 = None

      p1 = r1['p'][r1['rbit']]
      q1 = r1['p'][1-r1['rbit']]
      bc1 = None

      for bc in BS:
        ophk = block.choice(bc, 0)
        n = path.nelems(ophk)
        for k in range(n):
          omvk = path.elem(ophk, k)
          mvk, drk = move.unpack(omvk)
          if mvk.pt == (p0, q0): 
            bc0 = [bc, k]
          if mvk.pt == (p1, q1): 
            bc1 = [bc, k]

      if bc0 != None and bc1 != None:
        if bc0[1] > bc1[1]:
          ct = raster_raster_contact(bc0, bc1, parms)
        else:
          ct = raster_raster_contact(bc0, bc1, parms)
        CS.append(ct)

  
  print("# blocks:", len(BS))
  print("# contacts:", len(CS))
  return BS, CS

################################

wdfill = parms['solid_raster_width']

def test1():
  test_tiny("test1")

def test2():
  n = 5
  o = (1,1)
  test_road_of_rasters(n, o, "test2")

def test3():
  n = 6
  o = (1,1+(n-1)/2*wdfill)
  test_road_of_rasters(n, o, "test3")

def test4():
  nx = 3
  ny = 2
  nr = 7
  o = (1, 1)
  test_matrix_of_roads(nx, ny, nr, o, "test3")

def test5():
  nmv = 7
  nmg = 1
  nis = 1
  o = (1,1)
  Delta = 6
  maxcalls = 3000000
  test_two_roads_and_islands(nmv, nmg, nis, o, Delta, maxcalls, "test5")

def test6():
  nmv = 17
  nmg = 8
  nis = 2
  o = (1,1)
  Delta = 3.5
  maxcalls = 3000000
  test_two_roads_and_islands(nmv, nmg, nis, o, Delta, maxcalls, "test6")

def test7():
  sys.stderr.write("----------------------------------------------------------------------\n")
  #name = "hanger_0"
  #wdf = parms['solid_raster_width']

  name = "teste5"
  wdf = 0.50

  sys.stderr.write("--- %s ---\n" % name)

  parms['splitLines'] = True
  parms['limitLines'] = True
  parms['nMaxLines'] = 25
  parms['use_rasterlinks'] = True
  
  o = (0,0)
  
  BS, CS = create_input(name, wdf)

  input_data.plot(oprefix + name, BS, CS, o, wdf)
  os.remove(oprefix + name + '_ip00.eps')
  os.remove(oprefix + name + '_ip01.eps')
  
  Delta = 17
  maxcalls = 0
  
  Q = hotpath.best_path(o, BS, CS, Delta, maxcalls, parms)
  if Q != None:
    plot_solution(Q, name)  
  return 

# test1()
# test2()
# test3()
# test4()
# test5()
# test6()
test7()
